/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: timeout.c 
 * -----------------------------------------------------------------*/

/* --------------------------------------------------------------------------
 * BY: Stu Robinson.  Friday, November 21, 1997
 * ABSTRACT: This module defines functions to support timouts for port I/O
 *           All functions support the BESTTIMEOUTS structure declared in
 *           b_io.h (or COMMTIMEOUTS in windows) and are operating system
 *           independent EXCEPT:
 *           - OS must support a function that returns an incrementing counter.
 *
 * Some of the constructs here look a little strange ... they are
 * designed to be used by WinNT device drivers as well as application code.
 *
 * NOTES:    In accordance with the BESTTIMEOUTS structure, all timing is done
 *           in milliseconds.
 * -------------------------------------------------------------------------- */

#include "timeout.h"

/******************************************************************************/
/* OS dependent */

#if defined(_IA64_) && defined(DEVL)
LARGE_INTEGER ULONG_2_TICK(ULONG x) {
    LARGE_INTEGER y;
    y.QuadPart=x;
    return y;
}
#endif /* _ULONG2TICK_ */


/******************************************************************************/
/* these functions are driven by macros in timeouts.h */

static void BestGetOsTicks(b_ticktype * pTicks)
{
  GET_TICK_COUNT(pTicks);
  return;
}
/* --------------------------------------------------------------------------
 * Note that we'll never return 0 ticks from this function.
 * We also try to avoid overflow by doing some long-winded math.
 * -------------------------------------------------------------------------- */

static b_ticktype BestMs2Ticks(b_int32 Milliseconds)
{
  /* The real calculation...((Milliseconds * TICKS_PER_SEC) / 1000) */
  b_int32 uTps = TICKS_PER_SEC; /* this macro could be anything */
  b_int32 uTicks = ((Milliseconds / 1000) * uTps) + /* seconds */
  ((Milliseconds % 1000) * uTps) / 1000;  /* mS */
  return ULONG_2_TICK(uTicks ? uTicks : 1);
}


/******************************************************************************/
/* OS independent */

/* --------------------------------------------------------------------------
 * The first set of functions are all helper functions that have no knowledge
 * of the BESTTIMEOUTS structure.
 * We ALWAYS operate in ticks !
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * BestSetTicks() translates interval to ticks...internal to this file only
 * -------------------------------------------------------------------------- */

static void BestSetTicks(b_int32 IntervalInMs, PBEST_TIMER pBTS)
{
  /* capture the interval for fast restarts */
  pBTS->IntervalInMs = IntervalInMs;

  BestGetOsTicks(&(pBTS->StartTicks));
  pBTS->EndTicks = _ADD(pBTS->StartTicks, BestMs2Ticks(IntervalInMs));
  pBTS->fWrappedAround = (b_int8) _ISGT(pBTS->StartTicks, pBTS->EndTicks);
  return;
}


/* --------------------------------------------------------------------------
 * BestStartTimeout() can be called by anybody ... not for use w. BESTTIMEOUT
 * -------------------------------------------------------------------------- */
void BestStartTimeout(b_int32 IntervalInMs, PBEST_TIMER pBTS)
{
  /* MUST clear timer flags before setting it */
  CLEAR_TIMER(pBTS);
  BestSetTicks(IntervalInMs, pBTS);
}
/* --------------------------------------------------------------------------
 * This function checks any timeout for completion.
 * We need a mechanism for checking overflow (wrap-around) that works for
 * both signed and unsigned OS tick counters.
 * Note; the assumption is made that multiple-wrap-around is not an issue.
 * -------------------------------------------------------------------------- */

b_bool BestIsTimeoutDone(PBEST_TIMER pBTS)
{
  b_ticktype CurrTicks;
  /* note that if one of the special flags is set then the rest of the
   * structure may be invalid...we must check these FIRST */

  if (pBTS->fIsInfinite || pBTS->fWaitForRestart)
  {
    return 0;
  }
  else if (pBTS->fIsImmediate)
  {
    return 1;
  }

  /* it's a real timeout...get the "now" ticks */

  BestGetOsTicks(&CurrTicks);

  /* there are 4 cases that we have to be concerned with as far as overflow
   * is concerned...all combinations of either the EndTicks having wrapped
   * around or the CurrTicks having wrapped around. */

  if (pBTS->fWrappedAround)
  {
    /* current ticks MUST wrap around AND pass the end ticks */
    return _ISGT(pBTS->StartTicks, CurrTicks) &&
      _ISGT(CurrTicks, pBTS->EndTicks);
  }
  else
  {                             /* end ticks were not wrapped... */
    /* current ticks could either wrap around OR pass the end ticks */
    return _ISGT(pBTS->StartTicks, CurrTicks) ||
      _ISGT(CurrTicks, pBTS->EndTicks);
  }
}


/* --------------------------------------------------------------------------
 * Call BestInitTotalReadTimeout() BEFORE entering the reading loop.
 * Call BestIsTimeoutDone() AFTER a successful check of the interval timeout.
 * -------------------------------------------------------------------------- */

static void BestInitTotalReadTimeout(PBEST_TIMER pTmrTotal,
    LPBESTTIMEOUTS pCommTimeouts,
    b_int32 NumBytes)
{
  b_int32 TotalTimeoutInMs;
  if (pTmrTotal && pCommTimeouts)
  {
    /* MUST init every time */
    CLEAR_TIMER(pTmrTotal);

    /* ReadTotalTimeoutMultiplier == 0 && ReadTotalTimeoutConstant == 0
     * indicates special stuff.... */

    if ((0 == pCommTimeouts->ReadTotalTimeoutMultiplier) &&
      (0 == pCommTimeouts->ReadTotalTimeoutConstant))
    {
      /* pay attention here...in line with Windows we allow 2 options */

      if (B_INT32MAX == pCommTimeouts->ReadIntervalTimeout)
      {
        /* total timeout ends the first time that you check it !!!! */
        pTmrTotal->fIsImmediate = 1;
      }
      else
      {
        /* total timeout is not used (infinite). */
        pTmrTotal->fIsInfinite = 1;
      }
    }
    else
    {
      /* one more special case */
      if (B_INT32MAX == pCommTimeouts->ReadTotalTimeoutMultiplier)
        pCommTimeouts->ReadTotalTimeoutMultiplier = 0;

      /* set a normal timeout */
      TotalTimeoutInMs = (pCommTimeouts->ReadTotalTimeoutMultiplier * NumBytes) +
        pCommTimeouts->ReadTotalTimeoutConstant;

      BestSetTicks(TotalTimeoutInMs, pTmrTotal);
    }
  }
  return;
}


/* --------------------------------------------------------------------------
 * Call BestInitIntervalReadTimeout() BEFORE each read of the port.
 * Call BestIsTimeoutDone() AFTER a non-productive read to check if
 * that read has timed out.  Do NOT call it if the read succeeded !!
 * Return with as many bytes as you've already read if you timed out.
 *
 * NOTE; Pay special attention to the meaning of 0 and B_INT32MAX in the
 *       BESTTIMEOUTS structure !!!
 * -------------------------------------------------------------------------- */

static void BestInitIntervalReadTimeout(PBEST_TIMER pTmrInterval,
    LPBESTTIMEOUTS pCommTimeouts)
{
  if (pTmrInterval && pCommTimeouts)
  {
    /* MUST init every time */
    CLEAR_TIMER(pTmrInterval);

    /* interval timeout starts after the first char */
    pTmrInterval->fWaitForRestart = 1;

    /* This is screwy but....PAY ATTENTION; ReadIntervalTimeout == 0 is the
     * same as infinite ReadIntervalTimeout == B_INT32MAX is the same as
     * immediate */

    if (0 == pCommTimeouts->ReadIntervalTimeout)
    {
      /* timeout is not used (infinite). */
      pTmrInterval->fIsInfinite = 1;
    }
    else if (B_INT32MAX == pCommTimeouts->ReadIntervalTimeout)
    {
      /* timeout ends the first time that you check it !!!! */
      pTmrInterval->fIsImmediate = 1;
    }
    else
    {
      BestSetTicks(pCommTimeouts->ReadIntervalTimeout, pTmrInterval);
    }
  }
  return;
}


/******************************************************************************/
/* functions to initialize timers using the BESTTIMEOUTS structure */

/* --------------------------------------------------------------------------
 * Call BestInitReadTimeouts() BEFORE entering the reading loop.
 * Call BestIsReadTimeoutDone() AFTER each UNSUCCESSFUL read.
 * -------------------------------------------------------------------------- */

void BestInitReadTimeouts(PBEST_TIMER pTmrInterval,
    PBEST_TIMER pTmrTotal,
    LPBESTTIMEOUTS pUsrTimeouts,
    b_int32 NumBytes)
{
  BestInitIntervalReadTimeout(pTmrInterval, pUsrTimeouts);
  BestInitTotalReadTimeout(pTmrTotal, pUsrTimeouts, NumBytes);
}
/* --------------------------------------------------------------------------
 * Call BestInitWriteTimeout() BEFORE entering the writing loop.
 * Call BestIsWriteTimeoutDone() AFTER each UNSUCCESSFUL write.
 * -------------------------------------------------------------------------- */

void BestInitWriteTimeout(PBEST_TIMER pTmrTotal,
    LPBESTTIMEOUTS pCommTimeouts,
    b_int32 NumBytes)
{
  b_int32 TotalTimeoutInMs;
  if (pTmrTotal && pCommTimeouts)
  {
    /* MUST init every time */
    CLEAR_TIMER(pTmrTotal);

    /* check for special cases */
    if ((0 == pCommTimeouts->WriteTotalTimeoutMultiplier) &&
      (0 == pCommTimeouts->WriteTotalTimeoutConstant))
    {
      /* total timeout is not used (infinite). */
      pTmrTotal->fIsInfinite = 1;
    }
    else
    {
      /* one more special case */
      if (B_INT32MAX == pCommTimeouts->WriteTotalTimeoutMultiplier)
        pCommTimeouts->WriteTotalTimeoutMultiplier = 0;

      /* set a normal timeout */
      TotalTimeoutInMs = (pCommTimeouts->WriteTotalTimeoutMultiplier * NumBytes) +
        pCommTimeouts->WriteTotalTimeoutConstant;

      BestSetTicks(TotalTimeoutInMs, pTmrTotal);
    }
  }
  return;
}


/* --------------------------------------------------------------------------
 * This must be called AFTER an UNSUCCESSFUL read/write.
 * Note that, when reading, the Total timeout check is performed if
 * the interval timeout is not complete.
 * -------------------------------------------------------------------------- */
b_bool BestIsReadTimeoutDone(PBEST_TIMER pInterval, PBEST_TIMER pTotal)
{
  return (BestIsTimeoutDone(pInterval) || BestIsTimeoutDone(pTotal));
}

b_bool BestIsWriteTimeoutDone(PBEST_TIMER pTotal)
{
  return BestIsTimeoutDone(pTotal);
}
/* --------------------------------------------------------------------------
 * BestRestartIntervalTimeout() restarts a read-interval timeout.
 * It MUST be called anytime AFTER the first successful read (!!!!)
 * You MUST STILL call BestStartxxxTimeout() first !!
 * -------------------------------------------------------------------------- */

void BestRestartIntervalTimeout(PBEST_TIMER pTmrInterval)
{
  pTmrInterval->fWaitForRestart = 0;
  BestSetTicks(pTmrInterval->IntervalInMs, pTmrInterval);
  return;
}
/* --------------------------------------------------------------------------
 * BestSetCommToStruct() is exposed for easily filling a structure
 * -------------------------------------------------------------------------- */

void BestSetCommToStruct(LPBESTTIMEOUTS pCommTimeouts,
    b_int32 ReadIntervalTimeout,
    b_int32 ReadTotalTimeoutMultiplier,
    b_int32 ReadTotalTimeoutConstant,
    b_int32 WriteTotalTimeoutMultiplier,
    b_int32 WriteTotalTimeoutConstant)
{
  pCommTimeouts->ReadIntervalTimeout = ReadIntervalTimeout;
  pCommTimeouts->ReadTotalTimeoutMultiplier = ReadTotalTimeoutMultiplier;
  pCommTimeouts->ReadTotalTimeoutConstant = ReadTotalTimeoutConstant;
  pCommTimeouts->WriteTotalTimeoutMultiplier = WriteTotalTimeoutMultiplier;
  pCommTimeouts->WriteTotalTimeoutConstant = WriteTotalTimeoutConstant;
  return;
}
